//*************************************************************************************************
//
//	Description:
//		GhostingImposter.fx - Shader for renedering ghosting vehicle imposters on screne.
//
//	<P> Copyright (c) 2008 Blimey! Games Ltd. All rights reserved.
//
//*************************************************************************************************

#include "stddefs.fxh"

//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Transformation matrices
//

// View & Projection

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif


// Object-space into world space matrix.

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;


//
// Tweaking parameters.
//

float ghostDensity
<
	string UIName = "Ghost Density";
	bool appEdit = true;
> = 1.0f;


float ghostDistortion
<
	string UIName = "Ghost Distortion";
	bool appEdit = true;
> = 0.0f;


// float currentTime : TIME
//<
//	string UIWidget = "None";
//	bool appEdit = false;
//	bool export = false;
//> = 0.0f;


//
// Textures
//

// Imposter.

texture imposterTexture : TEXTURE
<
	string UIName = "Imposter Texture";
	bool appEdit = true;
>;


//
// Samplers
//

// Imposter.

sampler2D imposterMap : SAMPLER
<
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture = "imposterTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < imposterTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = None;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


//-----------------------------------------------------------------------
//
// Vertex Shader
//

// Input structure.

struct VSINPUT
{
	float3 position : POSITION;											// Object space position.
	float2 texCoord : TEXCOORD0;										// UV coords for imposter texture.
};


// Output structure.

struct VSOUTPUT
{
	float4 position	: POSITION;											// View-coords position.
	float2 imposter : TEXCOORD0;										// UV coords for imposter texture.
};


// Main vertex shader.

VSOUTPUT GhostVertexShader( VSINPUT input )
{
	VSOUTPUT output;

	// Calculate clip-space position of the vertex

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4 worldviewproj = mul( world, viewproj );
#endif
	output.position = mul( float4( input.position, 1.0f ), worldviewproj );

	// Pass UVs through.

	output.imposter = input.texCoord;

	return output;
}


//-----------------------------------------------------------------------
//
// Fragment Shaders
//

float4 GhostFragmentShader( VSOUTPUT input ) : COLOR0
{
	float4 imposterTexel = tex2D( imposterMap, input.imposter );

	// Highly distored ghost is greyscale - lerp between full colour version and greyscale version.

	float3 original = imposterTexel.rgb;
	float3 grey = dot( original, float3( 0.3f, 0.59f, 0.11f ) );
	float3 distortedColour = lerp( original, grey, ghostDistortion );

	float opacity = imposterTexel.a * ghostDensity;
	float3 colour = distortedColour * ghostDensity;

	return float4( colour, opacity );
}




//-----------------------------------------------------------------------
//
// Technique
//

technique Ghost
<
	bool supportsSpecialisedLighting = false;
  bool preservesGlobalState = true;
	string normalBehaviour	= "ERMB_RENDER";
	string normalTechnique	= "Ghost";
	int    normalDeferredID	= 2;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "ONE";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = ONE;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx GhostVertexShader();
		PixelShader = compile sce_fp_rsx GhostFragmentShader();
#else
		VertexShader = compile vs_3_0 GhostVertexShader();
		PixelShader = compile ps_3_0 GhostFragmentShader();
#endif
	}
}
